// ========================================
// MODE ARTISTIQUE D3 - VERSION TV RÉTRO
// Graphique radial avec présentation TV
// ========================================

let artisticSvg = null;
let currentArtisticYear = '2019';

function initArtisticD3() {
    console.log('Initialisation mode artistique D3');
    
    const container = d3.select('#artistic-d3-container');
    if (container.empty()) {
        console.error('Conteneur non trouvé');
        return;
    }
    
    // Nettoyer
    container.selectAll('svg, .artistic-info, .artistic-legend, .artistic-tv-indicator').remove();
    
    // Récupérer les dimensions du conteneur
    const containerNode = container.node();
    const width = containerNode.clientWidth || window.innerWidth;
    const height = containerNode.clientHeight || window.innerHeight - 80;
    
    console.log('Dimensions conteneur:', width, 'x', height);
    
    // Créer SVG avec les dimensions correctes
    artisticSvg = container.append('svg')
        .attr('id', 'artistic-svg')
        .attr('width', '100%')
        .attr('height', '100%')
        .attr('viewBox', `0 0 ${width} ${height}`);
    
    // Créer le graphique radial
    createRadialChart(artisticSvg, width, height);
    
    // Ajouter l'indicateur TV
    addTVIndicator(container);
    
    // Ajouter la légende
    addArtisticLegend(container);
    
    console.log('Mode artistique initialisé');
}

function createRadialChart(svg, width, height) {
    console.log('Création du graphique radial:', width, 'x', height);
    
    const centerX = width / 2;
    const centerY = height / 2;
    const maxRadius = Math.min(width, height) / 2 - 60;
    const innerRadius = Math.min(60, maxRadius * 0.3);
    
    const g = svg.append('g')
        .attr('transform', `translate(${centerX}, ${centerY})`);
    
    // Couleurs plus visibles
    const colors = {
        low: '#2b6cb0',
        medium: '#2f855a',
        high: '#c53030'
    };
    
    // Échelle
    const radiusScale = d3.scaleLinear()
        .domain([0, 100])
        .range([innerRadius, maxRadius]);
    
    // Couleurs
    const colorScale = d3.scaleLinear()
        .domain([30, 50, 70])
        .range([colors.low, colors.medium, colors.high]);
    
    // Fond du graphique
    g.append('circle')
        .attr('r', maxRadius + 20)
        .attr('fill', '#f0f4f8');
    
    // Anneaux concentriques
    [25, 50, 75, 100].forEach(level => {
        g.append('circle')
            .attr('r', radiusScale(level))
            .attr('fill', 'none')
            .attr('stroke', '#718096')
            .attr('stroke-width', 2);
        
        g.append('text')
            .attr('x', 5)
            .attr('y', -radiusScale(level) + 5)
            .attr('fill', '#2d3748')
            .attr('font-size', '12px')
            .attr('font-weight', 'bold')
            .attr('font-family', 'monospace')
            .text(level + '%');
    });
    
    // Cercle central
    g.append('circle')
        .attr('r', innerRadius)
        .attr('fill', '#fff')
        .attr('stroke', '#4a5568')
        .attr('stroke-width', 3);
    
    // Texte central
    g.append('text')
        .attr('text-anchor', 'middle')
        .attr('y', -5)
        .attr('fill', '#1a202c')
        .attr('font-size', '20px')
        .attr('font-weight', 'bold')
        .text('24H');
    
    g.append('text')
        .attr('text-anchor', 'middle')
        .attr('y', 18)
        .attr('fill', '#4a5568')
        .attr('font-size', '14px')
        .text('PARITÉ');
    
    // Données pour 24 heures
    const hours = d3.range(24).map(h => ({
        hour: h,
        value: 35 + Math.random() * 35
    }));
    
    // Barres radiales
    const angleStep = (2 * Math.PI) / 24;
    
    hours.forEach((d, i) => {
        const angle = i * angleStep - Math.PI / 2;
        const barLength = radiusScale(d.value) - innerRadius;
        
        // Ligne de référence
        g.append('line')
            .attr('x1', innerRadius * Math.cos(angle))
            .attr('y1', innerRadius * Math.sin(angle))
            .attr('x2', maxRadius * Math.cos(angle))
            .attr('y2', maxRadius * Math.sin(angle))
            .attr('stroke', '#a0aec0')
            .attr('stroke-width', 1);
        
        // Label heure
        const labelRadius = maxRadius + 25;
        g.append('text')
            .attr('x', labelRadius * Math.cos(angle))
            .attr('y', labelRadius * Math.sin(angle))
            .attr('text-anchor', 'middle')
            .attr('dominant-baseline', 'middle')
            .attr('fill', d.hour === 0 ? '#0066aa' : '#2d3748')
            .attr('font-size', '12px')
            .attr('font-weight', 'bold')
            .attr('font-family', 'monospace')
            .text(d.hour);
        
        // Barre
        const barWidth = Math.max(4, (maxRadius - innerRadius) / 15);
        g.append('rect')
            .attr('x', -barWidth / 2)
            .attr('y', innerRadius)
            .attr('width', barWidth)
            .attr('height', 0)
            .attr('fill', colorScale(d.value))
            .attr('rx', 3)
            .attr('transform', `rotate(${(angle * 180 / Math.PI) + 90})`)
            .transition()
            .duration(800)
            .delay(i * 30)
            .attr('height', barLength);
    });
    
    console.log('Graphique radial créé avec succès');
    
    // Info
    const avgValue = d3.mean(hours, d => d.value).toFixed(1);
    container.append('div')
        .attr('class', 'artistic-info')
        .html(`<strong style="color:#0066aa">Moyenne:</strong> ${avgValue}%`);
}

function addTVIndicator(container) {
    container.append('div')
        .attr('class', 'artistic-tv-indicator')
        .html(`
            <span class="led"></span>
            <span>MODE ART</span>
        `);
}

function addArtisticLegend(container) {
    const legend = container.append('div')
        .attr('class', 'artistic-legend');
    
    const legendData = [
        { label: 'Faible', color: '#2b6cb0' },
        { label: 'Moyen', color: '#2f855a' },
        { label: 'Élevé', color: '#c53030' }
    ];
    
    legendData.forEach(item => {
        const lg = legend.append('div')
            .attr('class', 'artistic-legend-item');
        
        lg.append('div')
            .attr('class', 'artistic-legend-color')
            .style('background-color', item.color);
        
        lg.append('span')
            .text(item.label);
    });
}

function updateArtisticYear(year) {
    currentArtisticYear = year;
    initArtisticD3();
}

function refreshArtisticD3() {
    initArtisticD3();
}

// Export
window.initArtisticD3 = initArtisticD3;
window.updateArtisticYear = updateArtisticYear;
window.refreshArtisticD3 = refreshArtisticD3;

